unit Mscheckb;

interface

uses
  Controls, Buttons, Classes, WinTypes, WinProcs, Messages, Graphics, Menus,
  ExtCtrls;

type
  TBitBtnNoBtn = class(TBitBtn)
  private
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
  end;

type

{TMultyStateCheckBox}

{An improved TBitBtn to continually change displayed glyphs by user click.
- Allows Implementation of custom check box with multiple states and signs.
- Supports multiple hints for each glyph.

Interface:

Additionally to mostly published or public TBitBtn fields there are:

NumGlyphs: Integer - a number of equal width glyphs within assigned Glyph
StateNum: Integer - current glyph number (0..NumGlyphs - 1)
HintItems: TStrings - a list of items (will be shown according to the StateNum)
UseHintItems: Boolean - use or not HintItems. Otherwise Hint string will be used.
}

  TMultyStateCheckBox = class(TCustomControl)
    private
      MPanel:         TPanel;
      SpeedButton:    TBitBtnNoBtn;
      FBitmap:        TBitmap;
      FExtNumGlyphs:  Integer;
      FStateNum:      Integer;
      FOnClick:       TNotifyEvent;
      FHintItems:     TStrings;
      FUseHintItems:  Boolean;
      FOnKeyDown:     TKeyEvent;
      FPopupShortCut: TShortCut;

      function GetExtGlyph: TBitmap;
      procedure SetExtGlyph(Picture: TBitmap);
      procedure SetExtNumGlyphs(newNum: Integer);
      procedure SetStateNum(newStateNum: Integer);
      function GetCaption:String;
      procedure SetCaption(newCaption: String);
      function GetLayout:TButtonLayout;
      procedure SetLayout(newLayout: TButtonLayout);

      procedure BtnClick(Sender: TObject);
      procedure KeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
{$IFNDEF VER80}
	    reintroduce;
{$ENDIF}

      function GetOnDblClick: TNotifyEvent;
      procedure SetOnDblClick(AEvent: TNotifyEvent);

      function GetOnDragDrop: TDragDropEvent;
      procedure SetOnDragDrop(AEvent: TDragDropEvent);

      function GetOnDragOver: TDragOverEvent;
      procedure SetOnDragOver(AEvent: TDragOverEvent);

      function GetOnEndDrag: TEndDragEvent;
      procedure SetOnEndDrag(AEvent: TEndDragEvent);

      function GetOnEnter: TNotifyEvent;
      procedure SetOnEnter(AEvent: TNotifyEvent);

      function GetOnExit: TNotifyEvent;
      procedure SetOnExit(AEvent: TNotifyEvent);

      function GetOnKeyPress: TKeyPressEvent;
      procedure SetOnKeyPress(AEvent: TKeyPressEvent);

      function GetOnKeyUp: TKeyEvent;
      procedure SetOnKeyUp(AEvent: TKeyEvent);

      function GetOnMouseDown: TMouseEvent;
      procedure SetOnMouseDown(AEvent: TMouseEvent);

      function GetOnMouseMove: TMouseMoveEvent;
      procedure SetOnMouseMove(AEvent: TMouseMoveEvent);

      function GetOnMouseUp: TMouseEvent;
      procedure SetOnMouseUp(AEvent: TMouseEvent);

      function GetHintItems: TStrings;
      procedure SetHintItems(AStrings: TStrings);

      procedure SetUseHintItems(Value: Boolean);
      procedure SetCurrentHint;
    protected
    public
      constructor Create(Owner: TComponent); override;
      destructor Destroy; override;
      function Focused: Boolean;
{$IFNDEF VER80}
	    override;
{$ENDIF}

    published

      property ParentShowHint;
      property PopupMenu;
      property ShowHint;
      property TabOrder;
      property Enabled;
      property Visible;

      property Caption: String read GetCaption write SetCaption;
      property Gliph: TBitmap read GetExtGlyph write SetExtGlyph;
      property NumGlyphs: Integer read FExtNumGlyphs write SetExtNumGlyphs;
      property StateNum: Integer read FStateNum write SetStateNum;
      property Layout: TButtonLayout read GetLayout write SetLayout;
      property HintItems: TStrings read GetHintItems write SetHintItems;
      property UseHintItems: Boolean read FUseHintItems write SetUseHintItems;
      property PopupShortCut: TShortCut read FPopupShortCut write FPopupShortCut; 

      property OnClick: TNotifyEvent read FOnClick write FOnClick;
      property OnDblClick: TNotifyEvent read GetOnDblClick write SetOnDblClick;
      property OnDragDrop: TDragDropEvent read GetOnDragDrop write SetOnDragDrop;
      property OnDragOver: TDragOverEvent read GetOnDragOver write SetOnDragOver;
      property OnEndDrag: TEndDragEvent read GetOnEndDrag write SetOnEndDrag;
      property OnEnter: TNotifyEvent read GetOnEnter write SetOnEnter;
      property OnExit: TNotifyEvent read GetOnExit write SetOnExit;
      property OnKeyDown: TKeyEvent read FOnKeyDown write FOnKeyDown;
      property OnKeyPress: TKeyPressEvent read GetOnKeyPress write SetOnKeyPress;
      property OnKeyUp: TKeyEvent read GetOnKeyUp write SetOnKeyUp;
      property OnMouseDown: TMouseEvent read GetOnMouseDown write SetOnMouseDown;
      property OnMouseMove: TMouseMoveEvent read GetOnMouseMove write SetOnMouseMove;
      property OnMouseUp: TMouseEvent read GetOnMouseUp write SetOnMouseUp;
  end;

implementation

{TBitBtnNoBtn}
procedure TBitBtnNoBtn.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_HASSETSEL + DLGC_WANTMESSAGE;
end;

{TMultyStateCheckBox}
constructor TMultyStateCheckBox.Create(Owner: TComponent);
begin
  inherited Create(Owner);
  FHintItems := TStringList.Create;
  FUseHintItems := False;

  FBitmap := TBitmap.Create;
  FExtNumGlyphs := 0;
  FStateNum := 0;
  MPanel := TPanel.Create(Self);
  MPanel.BevelOuter := bvLowered;
  MPanel.Align := alClient;
  MPanel.Parent := Self;
  MPanel.Visible := True;

  SpeedButton := TBitBtnNoBtn.Create(MPanel);
  SpeedButton.Parent := MPanel;
  SpeedButton.OnClick := BtnClick;
  SpeedButton.OnKeyDown := KeyDown;
  SpeedButton.Visible := True;
  SpeedButton.Enabled := True;
  SpeedButton.NumGlyphs := 1;
  SpeedButton.Glyph := FBitmap;
  SpeedButton.Align := alClient;

  Height := 22;
  Width := 22;
end;

destructor TMultyStateCheckBox.Destroy;
begin
  FHintItems.Destroy;
  SpeedButton.Destroy;
  FBitmap.Destroy;
  MPanel.Destroy;
  inherited Destroy;
end;

function TMultyStateCheckBox.GetExtGlyph: TBitmap;
begin
  Result := FBitmap;
end;

procedure TMultyStateCheckBox.SetExtGlyph(Picture: TBitmap);
begin
  FBitmap.Assign(Picture);
  if (FBitmap.Height = 0) or (FBitmap.Width = 0) then
  begin
    FExtNumGlyphs := 0;
    SpeedButton.Glyph := FBitmap;
  end
  else
  begin
    FExtNumGlyphs := FBitmap.Width div FBitmap.Height;
    if (FBitmap.Width mod FBitmap.Height) > 0 then
      FExtNumGlyphs := FExtNumGlyphs + 1;
    SetStateNum(0);
  end;
end;

procedure TMultyStateCheckBox.SetExtNumGlyphs(newNum: Integer);
begin
  if newNum > 0 then
  begin
    FExtNumGlyphs := newNum;
    SetStateNum(0);
  end;
end;

procedure TMultyStateCheckBox.SetStateNum(newStateNum: Integer);
begin
  if FExtNumGlyphs <= 0 then
  begin
    FStateNum := 0;
    Exit;
  end;

  FStateNum := newStateNum;
  SpeedButton.Glyph := FBitmap;
  SpeedButton.Glyph.Width := FBitmap.Width div FExtNumGlyphs;
  SpeedButton.Glyph.Height := FBitmap.Height;
  if FExtNumGlyphs <> 0 then
    BitBlt(SpeedButton.Glyph.Canvas.Handle, 0, 0,
      FBitmap.Width div FExtNumGlyphs, FBitmap.Height, FBitmap.Canvas.Handle,
      (FBitmap.Width div FExtNumGlyphs) * newStateNum, 0, SRCCOPY);

  SpeedButton.NumGlyphs := 1;
  SpeedButton.Style := bsNew;

  SetCurrentHint;
end;

procedure TMultyStateCheckBox.SetCurrentHint;
begin
  if FUseHintItems and (HintItems.Count = FExtNumGlyphs) then
  begin
    ShowHint := True;
    Hint := HintItems[FStateNum];
  end;
end;

function TMultyStateCheckBox.GetCaption:String;
begin
  Result := SpeedButton.Caption;
end;

procedure TMultyStateCheckBox.SetCaption(newCaption: String);
begin
  SpeedButton.Caption := newCaption;
end;

function TMultyStateCheckBox.GetLayout:TButtonLayout;
begin
  Result := SpeedButton.Layout;
end;

procedure TMultyStateCheckBox.SetLayout(newLayout: TButtonLayout);
begin
  SpeedButton.Layout := newLayout;
end;

procedure TMultyStateCheckBox.BtnClick(Sender: TObject);
begin
  if FExtNumGlyphs > 0 then
  begin
    if FStateNum < FExtNumGlyphs - 1 then
      SetStateNum(FStateNum + 1)
    else
      SetStateNum(0);
  end;
  if Assigned(FOnClick) then
    FOnClick(Self);
end;

function TMultyStateCheckBox.Focused: Boolean;
begin
  Result := SpeedButton.Focused;
end;

function TMultyStateCheckBox.GetOnDblClick: TNotifyEvent;
begin
  Result := SpeedButton.OnDblClick;
end;

procedure TMultyStateCheckBox.SetOnDblClick(AEvent: TNotifyEvent);
begin
  SpeedButton.OnDblClick := AEvent;
end;

function TMultyStateCheckBox.GetOnDragDrop: TDragDropEvent;
begin
  Result := SpeedButton.OnDragDrop;
end;

procedure TMultyStateCheckBox.SetOnDragDrop(AEvent: TDragDropEvent);
begin
  SpeedButton.OnDragDrop := AEvent;
end;

function TMultyStateCheckBox.GetOnDragOver: TDragOverEvent;
begin
  Result := SpeedButton.OnDragOver;
end;

procedure TMultyStateCheckBox.SetOnDragOver(AEvent: TDragOverEvent);
begin
  SpeedButton.OnDragOver := AEvent;
end;

function TMultyStateCheckBox.GetOnEndDrag: TEndDragEvent;
begin
  Result := SpeedButton.OnEndDrag;
end;

procedure TMultyStateCheckBox.SetOnEndDrag(AEvent: TEndDragEvent);
begin
  SpeedButton.OnEndDrag := AEvent;
end;

function TMultyStateCheckBox.GetOnEnter: TNotifyEvent;
begin
  Result := SpeedButton.OnEnter;
end;

procedure TMultyStateCheckBox.SetOnEnter(AEvent: TNotifyEvent);
begin
  SpeedButton.OnEnter := AEvent;
end;

function TMultyStateCheckBox.GetOnExit: TNotifyEvent;
begin
  Result := SpeedButton.OnExit;
end;

procedure TMultyStateCheckBox.SetOnExit(AEvent: TNotifyEvent);
begin
  SpeedButton.OnExit := AEvent;
end;

function TMultyStateCheckBox.GetOnKeyPress: TKeyPressEvent;
begin
  Result := SpeedButton.OnKeyPress;
end;

procedure TMultyStateCheckBox.SetOnKeyPress(AEvent: TKeyPressEvent);
begin
  SpeedButton.OnKeyPress := AEvent;
end;

function TMultyStateCheckBox.GetOnKeyUp: TKeyEvent;
begin
  Result := SpeedButton.OnKeyUp;
end;

procedure TMultyStateCheckBox.SetOnKeyUp(AEvent: TKeyEvent);
begin
  SpeedButton.OnKeyUp := AEvent;
end;

function TMultyStateCheckBox.GetOnMouseDown: TMouseEvent;
begin
  Result := SpeedButton.OnMouseDown;
end;

procedure TMultyStateCheckBox.SetOnMouseDown(AEvent: TMouseEvent);
begin
  SpeedButton.OnMouseDown := AEvent;
end;

function TMultyStateCheckBox.GetOnMouseMove: TMouseMoveEvent;
begin
  Result := SpeedButton.OnMouseMove;
end;

procedure TMultyStateCheckBox.SetOnMouseMove(AEvent: TMouseMoveEvent);
begin
  SpeedButton.OnMouseMove := AEvent;
end;

function TMultyStateCheckBox.GetOnMouseUp: TMouseEvent;
begin
  Result := SpeedButton.OnMouseUp;
end;

procedure TMultyStateCheckBox.SetOnMouseUp(AEvent: TMouseEvent);
begin
  SpeedButton.OnMouseUp := AEvent;
end;

function TMultyStateCheckBox.GetHintItems: TStrings;
begin
  Result := FHintItems;
end;

procedure TMultyStateCheckBox.SetHintItems(AStrings: TStrings);
begin
  FHintItems.Assign(AStrings);
end;

procedure TMultyStateCheckBox.KeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var
  P:  TPoint;
begin
  if Assigned(FOnKeyDown) then
    FOnKeyDown(Self, Key, Shift);
  if FPopupShortCut = ShortCut(Key, Shift) then
  begin
    P.X := Width;
    P.Y := Height;
    P := ClientToScreen(P);
    if PopupMenu <> nil then
      PopupMenu.Popup(P.X, P.Y);
  end;
end;

procedure TMultyStateCheckBox.SetUseHintItems(Value: Boolean);
begin
  FUseHintItems := Value;
  if FUseHintItems then
  begin
    SetCurrentHint;
    ShowHint := True;
  end;
end;

end.
